<?php
/**
 * Class Hello_Sajto_Press_Releases
 *
 * Handles the admin page for the 'Press Releases' submenu.
 *
 * @package Hello_Sajto
 */
class Hello_Sajto_Press_Releases {

	/**
	 * Render the admin page for 'Press Releases'.
	 */
	public static function render_page() {
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_die( __( 'You do not have sufficient permissions to access this page.', 'hello-sajto-for-news-sites' ) );
		}

		$api               = new Hello_Sajto_Call_API();
		$categories        = $api->call_api( 'get_categories' );
		$selected_category = isset( $_GET['category'] ) ? sanitize_text_field( $_GET['category'] ) : '';
		$search_keyword    = isset( $_GET['keyword'] ) ? sanitize_text_field( $_GET['keyword'] ) : '';
		$posts             = self::get_posts( $selected_category, $search_keyword );
		$items             = self::get_items( $posts );
		$list_table        = new Hello_Sajto_Press_Releases_List_Table( $items, $categories );

		include_once HELLO_SAJTO_PLUGIN_DIR . 'templates/admin/press-releases.php';
	}

	/**
	 * Get the posts based on the selected category.
	 *
	 * @param string $selected_category The selected category ID.
	 * @param string $search_keyword    The search keyword.
	 *
	 * @return array The posts data.
	 */
	private static function get_posts( string $selected_category = '', $search_keyword = '' ) {
		$query_args = array(
			'page'     => 1,
			'per_page' => 30,
			'after'    => ( new DateTime( 'now -6 month' ) )->format( 'Y-m-d\TH:i:s' ),
			'orderby'  => 'date',
			'order'    => 'desc',
		);

		if ( ! empty( $selected_category ) ) {
			$query_args['categories'] = $selected_category;
		}

		if ( ! empty( $search_keyword ) ) {
			$query_args['search'] = $search_keyword;
		}

		return ( new Hello_Sajto_Call_API() )->call_api( 'get_posts', $query_args );
	}

	/**
	 * Get the items for the list table.
	 *
	 * @param array $posts      The posts data.
	 *
	 * @return array The formatted items for the list table.
	 */
	private static function get_items( $posts ) {
		return array_map(
			function ( $post ) {
				return array(
					'cb'       => array(
						'id' => $post['id'],
					),
					'title'    => array(
						'id'             => $post['id'],
						'title'          => $post['title']['rendered'],
						'lead'           => $post['excerpt']['rendered'],
						'url'            => $post['link'],
						'date'           => date_i18n( get_option( 'date_format' ), strtotime( $post['date'] ) ),
						'author'         => $post['author_name'] ?? '',
						'images_num'     => $post['images_num'],
						'contains_video' => $post['contains_video'],
					),
					'category' => implode( ', ', $post['category_names'] ?? array() ),
					'tags'     => implode( ', ', $post['tag_names'] ?? array() ),
					'action'   => array(
						'id'  => $post['id'],
						'url' => $post['link'],
					),
				);
			},
			$posts
		);
	}

	/**
	 * Get the name of the category by its ID.
	 *
	 * @param array $post     The post data.
	 * @param array $categories  The list of categories.
	 *
	 * @return string The name of the category or 'Uncategorized' if not found.
	 */
	private static function get_post_category_name( $post, $categories ) {
		if ( empty( $post )
		|| empty( $post['categories'] )
		|| ! is_array( $post['categories'] )
		|| ! isset( $post['categories'][0] )
		|| empty( $categories )
		|| ! is_array( $categories ) ) {
			return __( 'Uncategorized', 'hello-sajto-for-news-sites' );
		}

		foreach ( $categories as $category ) {
			if ( (int) $category['id'] === (int) $post['categories'][0] ) {
				return $category['name'];
			}
		}

		return __( 'Uncategorized', 'hello-sajto-for-news-sites' );
	}
}
