<?php

if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Class Hello_Sajto_Press_Releases_List_Table
 *
 * Custom table to display press releases.
 *
 * @package Hello_Sajto
 */
class Hello_Sajto_Press_Releases_List_Table extends WP_List_Table {
	/**
	 * Items to display in the table.
	 *
	 * @var array
	 */
	public $items;

	/**
	 * Categories for filtering.
	 *
	 * @var array
	 */
	public $categories;

	/**
	 * Constructor.
	 *
	 * @param array $items Items to display in the table.
	 */
	public function __construct( $items = array(), $categories = array() ) {
		parent::__construct(
			array(
				'singular' => 'press_release',
				'plural'   => 'press_releases',
				'ajax'     => false,
			)
		);
		$this->items      = $items;
		$this->categories = $categories;
	}

	/**
	 * Get columns for the table.
	 *
	 * @return array Columns of the table.
	 */
	public function get_columns() {
		return array(
			'cb'       => '<input type="checkbox" />',
			'title'    => __( 'Title', 'hello-sajto-for-news-sites' ),
			'category' => __( 'Category', 'hello-sajto-for-news-sites' ),
			'tags'     => __( 'Tags', 'hello-sajto-for-news-sites' ),
			'action'   => '',
		);
	}

	/**
	 * Prepare items for display.
	 */
	public function prepare_items() {
		$this->_column_headers = array( $this->get_columns(), array(), array() );
		$this->items           = $this->items;
	}

	/**
	 * Default column rendering.
	 *
	 * @param array  $item        Item data.
	 * @param string $column_name Column name.
	 *
	 * @return string Column content.
	 */
	protected function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'cb':
			case 'title':
			case 'category':
			case 'tags':
			case 'action':
				return $item[ $column_name ];
			default:
				return '';
		}
	}

	/**
	 * Render the checkbox column.
	 *
	 * @param array $item The item data.
	 *
	 * @return string The checkbox HTML.
	 */
	protected function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="bulk-select[]" value="%d" />',
			esc_attr( $item['cb']['id'] )
		);
	}

	/**
	 * Render the title column with a link to the post.
	 *
	 * @param array $item The item data.
	 *
	 * @return string The formatted title with actions.
	 */
	protected function column_title( $item ) {
		$title     = isset( $item['title']['title'] ) ? $item['title']['title'] : '';
		$url       = isset( $item['title']['url'] ) ? $item['title']['url'] : '';
		$date      = isset( $item['title']['date'] ) ? '<div title="'
						. __( 'Publish date', 'hello-sajto-for-news-sites' ) . '">' . $item['title']['date'] . '</div>' : '';
		$author    = isset( $item['title']['author'] ) ? '<div title="'
						. __( 'Author', 'hello-sajto-for-news-sites' ) . '">' . $item['title']['author'] . '</div>' : '';
		$images    = ( $item['title']['images_num'] ? '<div><span class="dashicons dashicons-format-image" title="'
						. __( 'Image', 'hello-sajto-for-news-sites' ) . '"></span> '
						. esc_html( $item['title']['images_num'] ) . '</div>' : '' );
		$video     = ( $item['title']['contains_video'] ? '<div><span class="dashicons dashicons-format-video" title="'
						. __( 'Video', 'hello-sajto-for-news-sites' ) . '"></span></div>' : '' );
		$meta_info = '<div class="hs-post-meta-info">' . $date . $author . $images . $video . '</div>';
		$lead      = $item['title']['lead'] ? '<div class="hs-post-lead">' . strip_tags( $item['title']['lead'] ) . '</div>' : '';

		return '<div class="hs-post-title"><a href="' . $url . '" target="_blank">' . $title . '</a></div>' . $meta_info . $lead;
	}

	/**
	 * Render the action column with buttons.
	 *
	 * @param array $item The item data.
	 *
	 * @return string The formatted action buttons.
	 */
	protected function column_action( $item ) {
		$html = '<div class="hs-btn-group">'
		. sprintf(
			'<a href="%s" target="_blank" title="%s">%s</a>',
			esc_url( $item['action']['url'] ),
			esc_attr__( 'View', 'hello-sajto-for-news-sites' ),
			'<span class="dashicons dashicons-external"></span>'
		)
		. sprintf(
			'<input type="button" class="button button-primary hs-import-button" data-post-id="%d" value="%s" />',
			esc_attr( $item['action']['id'] ),
			__( 'Import', 'hello-sajto-for-news-sites' )
		)
		. '</div>';

		return $html;
	}

	/**
	 * Add custom content above or below the table.
	 *
	 * @param string $which The location of the extra table nav ('top' or 'bottom').
	 */
	protected function extra_tablenav( $which ) {
		if ( 'top' === $which ) {
			$total_items = count( $this->items );
			$keyword     = isset( $_GET['keyword'] ) ? sanitize_text_field( $_GET['keyword'] ) : '';

			// Filter.
			require_once HELLO_SAJTO_PLUGIN_DIR . 'templates/admin/form/filter.php';

			// Total items.
			echo '<div class="alignright">';
			/* translators: %d: Total number of items. */
			echo '<strong>' . esc_html( sprintf( __( 'Total: %d', 'hello-sajto-for-news-sites' ), $total_items ) ) . '</strong>';
			echo '</div>';
		}
	}

	/**
	 * Get bulk actions for the table.
	 *
	 * @return array Bulk actions.
	 */
	protected function get_bulk_actions() {
		return array(
			'import' => __( 'Import', 'hello-sajto-for-news-sites' ),
		);
	}
}
