<?php
/**
 * Class Hello_Sajto_Import
 *
 * @package Hello_Sajto
 */
class Hello_Sajto_Import {
	/**
	 * External post ID to import.
	 *
	 * @var int $external_post_id
	 */
	private int $external_post_id;

	/**
	 * Post data to import.
	 *
	 * @var array $post_data
	 */
	private array $post_data;

	/**
	 * Media data to import.
	 *
	 * @var array $media_data
	 */
	private array $media_data;

	/**
	 * Whether to import tags.
	 *
	 * @var bool $import_tags
	 */
	private bool $import_tags = false;

	/**
	 * Hello_Sajto_Import constructor.
	 *
	 * @param int $external_post_id The ID of the external post to import.
	 */
	public function __construct( $external_post_id ) {
		$this->external_post_id = $external_post_id;
		$this->import_tags      = (bool) get_option( 'hello_sajto_import_tags', false );
	}

	/**
	 * Import the post from the external API and insert it into WordPress.
	 *
	 * @return int|WP_Error The post ID on success, or a WP_Error object on failure.
	 * @throws Exception On error.
	 */
	public function import() {
		set_time_limit( 600 );

		$this->set_post_data();

		// Insert WP post based on WP REST API response.
		$post_id = wp_insert_post(
			array(
				'post_title'   => $this->post_data['title']['rendered'],
				'post_content' => $this->post_data['content']['rendered'],
				'post_excerpt' => wp_strip_all_tags( $this->post_data['excerpt']['rendered'] ),
				'post_status'  => 'draft',
				'post_type'    => 'post',
				'post_author'  => get_current_user_id(),
			)
		);

		if ( is_wp_error( $post_id ) ) {
			throw new Exception(
				sprintf(
					__( 'Failed to insert post with external ID %1$d: %2$s', 'hello-sajto-for-news-sites' ),
					$this->external_post_id,
					$post_id->get_error_message()
				)
			);
		}

		// Import tags if enabled.
		$this->import_tags( $post_id );

		// Set post media data.
		$this->set_media_data();

		// Import media associated with the post.
		$this->import_media_data( $post_id );

		return $post_id;
	}

	/**
	 * Import tags associated with the post.
	 *
	 * @param int $post_id The ID of the post to associate the tags with.
	 *
	 * @return void
	 */
	private function import_tags( int $post_id ) {
		if ( ! $this->import_tags || empty( $this->post_data['tag_names'] ) || ! is_array( $this->post_data['tag_names'] ) ) {
			return;
		}

		// Import tags associated with the post.
		foreach ( $this->post_data['tag_names'] as $tag_name ) {
			$term = term_exists( $tag_name, 'post_tag' );

			if ( $term ) {
				// Get the term and convert it to an array structure similar to wp_insert_term().
				$term_object = get_term( $term['term_id'], 'post_tag' );
				$tag         = array(
					'term_id'          => $term_object->term_id,
					'term_taxonomy_id' => $term_object->term_taxonomy_id,
				);
			} else {
				// Insert the term if it doesn't exist.
				$tag = wp_insert_term( $tag_name, 'post_tag' );
			}

			wp_set_post_terms( $post_id, array( $tag['term_id'] ), 'post_tag', true );
		}
	}

	/**
	 * Import the media associated with the post.
	 *
	 * @param int $post_id The ID of the post to associate the media with.
	 *
	 * @return void
	 * @throws Exception If the media data cannot be fetched or is invalid.
	 */
	private function import_media_data( int $post_id ) {
		if ( empty( $this->media_data ) ) {
			return;
		}

		// Import images which comes from /wp-json/wp/v2/media?parent=111 REST API endpoint.
		foreach ( $this->media_data as $media ) {
			if ( ! isset( $media['id'], $media['source_url'], $media['media_type'] )
				|| 'image' !== $media['media_type'] ) {
				continue;
			}

			$media_url     = $media['source_url'];
			$media_caption = isset( $media['caption']['rendered'] ) ? strip_tags( $media['caption']['rendered'] ) : '';

			// Sideload the image and get the attachment ID.
			$attachment_id = media_sideload_image( $media_url, $post_id, '', 'id' );

			if ( ! is_wp_error( $attachment_id ) ) {
				// Update the caption for the attachment.
				wp_update_post(
					array(
						'ID'           => $attachment_id,
						'post_excerpt' => $media_caption,
					)
				);
			}
		}
	}

	/**
	 * Get the post data from the external API.
	 *
	 * @return void
	 * @throws Exception If the post data cannot be fetched or is invalid.
	 */
	private function set_post_data(): void {
		$api  = new Hello_Sajto_Call_Api();
		$data = $api->call_api( 'get_posts', array( 'include[]' => $this->external_post_id ) );

		if ( is_wp_error( $data ) ) {
			throw new Exception(
				sprintf(
					__( 'Failed to fetch post data for external ID %1$d: %2$s', 'hello-sajto-for-news-sites' ),
					$this->external_post_id,
					$data->get_error_message()
				)
			);
		}

		if ( empty( $data ) || ! is_array( $data ) || ! isset( $data[0] ) ) {
			throw new Exception(
				sprintf(
					__( 'No post data found for external ID %1$d', 'hello-sajto-for-news-sites' ),
					$this->external_post_id
				)
			);
		}

		$this->post_data = $data[0];
	}

	/**
	 * Get the media data associated with the post from the external API.
	 *
	 * @return void
	 * @throws Exception If the media data cannot be fetched or is invalid.
	 */
	private function set_media_data(): void {
		$api  = new Hello_Sajto_Call_Api();
		$data = $api->call_api( 'get_media', array( 'parent' => $this->external_post_id ) );

		if ( is_wp_error( $data ) ) {
			throw new Exception(
				sprintf(
					__( 'Failed to fetch media data for external ID %1$d: %2$s', 'hello-sajto-for-news-sites' ),
					$this->external_post_id,
					$data->get_error_message()
				)
			);
		}

		if ( ! is_array( $data ) ) {
			throw new Exception(
				sprintf(
					__( 'Invalid media data found for external ID %1$d', 'hello-sajto-for-news-sites' ),
					$this->external_post_id
				)
			);
		}

		$this->media_data = $data;
	}
}
