<?php
/**
 * Class Hello_Sajto_Call_Api
 *
 * @package Hello_Sajto
 */
class Hello_Sajto_Call_Api {

	/**
	 * The API domain.
	 *
	 * @var string
	 */
	private const API_DOMAIN = 'hellosajto.hu';

	/**
	 * The API URL to call.
	 *
	 * @var string
	 */
	private string $url = '';

	/**
	 * The API endpoint to call.
	 *
	 * @var string
	 */
	private string $endpoint = '';

	/**
	 * Action / endpoints map.
	 *
	 * @var string
	 */
	private array $endpoints = array(
		'get_posts'      => '/wp/v2/posts',
		'get_categories' => '/wp/v2/categories',
		'get_media'      => '/wp/v2/media',
	);

	/**
	 * The username for the API.
	 *
	 * @var string
	 */
	private string $username = '';

	/**
	 * The password for the API.
	 *
	 * @var string
	 */
	private string $password = '';

	/**
	 * Hello_Sajto_Call_Api constructor.
	 */
	public function __construct() {
		// Load settings.
		$this->username = get_option( 'hello_sajto_api_username', '' );
		$this->password = get_option( 'hello_sajto_api_password', '' );

		if ( ! $this->user_is_set() ) {
			throw new Exception(
				__( 'Missing API username and / or password. You can set it on Settings > Helló Sajtó page.', 'hello-sajto-for-news-sites' )
			);
		}
	}

	/**
	 * Set the API username.
	 *
	 * @return bool
	 */
	public function user_is_set(): bool {
		return ! empty( $this->username ) && ! empty( $this->password );
	}

	/**
	 * Call the API and return the response.
	 *
	 * @return array|WP_Error The API response or WP_Error on failure.
	 */
	public function call_api( $action, $params = array() ) {
		$this->endpoint = $this->endpoints[ $action ] ?? null;

		if ( ! $this->endpoint ) {
			return new WP_Error( 'invalid_action', __( 'Invalid action specified.', 'hello-sajto-for-news-sites' ) );
		}

		$this->url = 'https://' . self::API_DOMAIN . '/wp-json' . $this->replace_placeholders( $this->endpoint, $params );

		// Encode username and password for Basic Authentication.
		$auth = base64_encode( $this->username . ':' . $this->password );

		$response = wp_remote_get(
			$this->url,
			array(
				'timeout' => 15,
				'headers' => array(
					'Accept'        => 'application/json',
					'Authorization' => 'Basic ' . $auth,
				),
			)
		);

		// Check if the response is valid.
		if ( 200 !== wp_remote_retrieve_response_code( $response ) ) {
			throw new Exception( __( 'Invalid HTTP response status code.', 'hello-sajto-for-news-sites' ) );
		}

		// Check for errors in the response.
		if ( is_wp_error( $response ) ) {
			throw new Exception( __( 'API response error: ', 'hello-sajto-for-news-sites' . $response->get_error_message() ) );
		}

		return json_decode( wp_remote_retrieve_body( $response ), true );
	}

	/**
	 * Replace placeholders in the endpoint with actual parameters.
	 *
	 * @param string $endpoint The endpoint with placeholders.
	 * @param array  $params   The parameters to replace in the endpoint.
	 *
	 * @return string The endpoint with replaced parameters.
	 */
	private function replace_placeholders( $endpoint, $params ) {
		if ( ! empty( $params ) ) {
			$query_string = http_build_query( $params );
			return $endpoint . '?' . $query_string;
		}
		return $endpoint;
	}
}
